/**
 * Communication Slave Module.
 * Sits in each page.
 * @author Yoel Gluschnaider
 */
var comSlave = (function () 
{
	
	var me = {};
	
	// The handshake message string.
	var HANDSHAKE_MSG = 'HANDSHAKE';
	
	// The ACK message string
	var HANDSHAKE_ACK_MSG = 'ACK';
	
	// the receive call back.
	var m_recCb;
		
	// an array of all the channels receivers.
	var m_recArr = [];
	
	// the final receiver
	var m_receiver;

	// The final sender of this slave.
	var m_sender;
	
	// the senders array to send the handshake
	var m_sendersArr = [];
	
	// The id of this slave (set in the handshake process).
	var m_id = null;
	

	/**
	 * Stops all listeners and clears the array.
	 * 
	 * @author Yoel Gluschnaider
	 */
	function stopAllReceivers() 
	{
		var recArrLen = m_recArr.length;
		for (var i = 0; i < recArrLen; i++) 
		{
			m_recArr[i].stop();
		}
		// clear out the array.
		m_recArr = [];
	}
	
	/**
	 * Message receive call back.
	 * @param {Object} messageObj
	 * 
	 * @author Yoel Gluschnaider
	 */
	function receiverCb(messageObj) 
	{
		if (m_recCb) 
		{
			m_recCb(messageObj.msg);
		}
	}
	
	/**
	 * A listener call back for the handshake event
	 * @param {Object} messageObj - the handshake message.
	 * 
	 * @author Yoel Gluschnaider
	 */
	function ackListener(messageObj) 
	{
		// sanity check
		if (messageObj.msg.messageTxt !== HANDSHAKE_ACK_MSG) 
		{
			return;
		}
		
		// get the id of the slave.
		m_id = messageObj.msg.slaveId;
		
		// Stop listening to all channels.
		stopAllReceivers();
		
		// check the message type so we will know which sender to create.
		switch (messageObj.type) 
		{
			case comTypeEnum.postMessage:
				m_sender = new PostMsgSender(messageObj.metaData);
				m_receiver = new PostMsgReceiver(receiverCb);
			break;
            case comTypeEnum.nativeMessage:
                m_sender = new NativeMsgSender(messageObj.metaData);
                m_receiver = new NativeMsgReceiver(receiverCb);
                break;
			case comTypeEnum.iframeMessage:
				m_sender = new IframeSender(parent);
				m_receiver = new IframeReceiver(receiverCb);
			break;
		}

		// notify the page we are ready.
		m_readyCb(m_id);
	}
	
	/**
	 * Creates receivers to all channels.
	 * 
	 * @author Yoel Gluschnaider
	 */
	function listenToAllChannels() 
	{
		m_recArr.push(new PostMsgReceiver(ackListener));
        m_recArr.push(new NativeMsgReceiver(ackListener));
		m_recArr.push(new IframeReceiver(ackListener));
	}
	
	/**
	 * Creates senders to all the channels
	 * 
	 * @author Yoel Gluschnaider
	 */
	function createSendersToAllChannels() 
	{
		// check that there is a parent before creating the post message sender.
		if (parent) 
		{
			m_sendersArr.push(new PostMsgSender(parent));
			m_sendersArr.push(new IframeSender(parent));
		}
        m_sendersArr.push(new NativeMsgSender('appbase'));
	}
	
	/**
	 * Sends handshake message to all the channels
	 * 
	 * @author Yoel Gluschnaider
	 */
	function sendHandshakes() 
	{
		createSendersToAllChannels();
		
		// send handshakes to all channels
		var handShakeMessage = {messageTxt: HANDSHAKE_MSG};
		var arrLen = m_sendersArr.length;
		for (var i=0; i < arrLen; i++) 
		{
			m_sendersArr[i].sendMessage(handShakeMessage);
		}

		// clear the senders array after sending handshakes.
		m_sendersArr = [];		
	}
	
	/**
	 * Register on the receive call back.
	 * @param {Function} receiveCb - the receive call back.
	 * 
	 * @author Yoel Gluschnaider
	 */
	me.registerOnReceive = function(receiveCb) 
	{
		m_recCb = receiveCb;
	};
		
	/**
	 * Initializes a new slave.
	 * @param {Function} readyCb - call back when ready (established connection with the comMaster).
	 * 
	 * @author Yoel Gluschnaider
	 */
	me.init = function (readyCb) 
	{
		m_readyCb = readyCb;
		listenToAllChannels();
		sendHandshakes();
	};
	
	/**
	 * Sends message to the master.
	 * Adds the slave id to the message.
	 * @param {Object} messageObj - the message to send
	 * 
	 * @author Yoel Gluschnaider
	 */
	me.sendMessage = function (messageObj) 
	{
		m_sender.sendMessage({
			msg: messageObj,
			slaveId: m_id
		});
	};
	
	return me;
	
})();
